/***************************************************************************
*
* Copyright 2010,2011 BMW Car IT GmbH
*
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*        http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
****************************************************************************/
#include "CommitCommand.h"
#include "ICommandExecutor.h"
#include "Scene.h"
#include "Log.h"
#include <string>

/*failing command to be printed to errmem
 * should only contain commands which could
 * potantialy lead to black screen issues*/
static std::string lm_command_req_errmem_entry[] =
{
        "ScreenSetRenderOrderCommand",
        "LayerAddSurfaceCommand",
        "LayerSetRenderOrderCommand",
        "LayerSetVisibilityCommand",
        "SurfaceSetVisibilityCommand"
};

bool needs_errmem_entry(std::string command_name)
{
    int i;
    int func_cnt = sizeof(lm_command_req_errmem_entry)/sizeof(lm_command_req_errmem_entry[0]);

    for (i = 0; i < func_cnt; i++)
    {
        if (lm_command_req_errmem_entry[i] == command_name)
            return true;
    }

    return false;
}
ExecutionResult CommitCommand::execute(ICommandExecutor* executor)
{
    bool success = true;
    bool redraw = false;

    unsigned int commitCommandPid = getSenderPid();

    CommandList& clientCommandQueue = executor->getEnqueuedCommands(commitCommandPid);
    CommandListIterator iter = clientCommandQueue.begin();
    CommandListIterator iterEnd = clientCommandQueue.end();

    for (; iter != iterEnd; ++iter)
    {
        ICommand* command = (*iter);
        if (command)
        {
            bool commandExecuteSuccess = true;
            ExecutionResult result = command->execute(executor);

            if (result == ExecutionFailed || result == ExecutionFailedRedraw)
            {
                success = false;
                commandExecuteSuccess = false;
            }

            if (result == ExecutionFailedRedraw || result == ExecutionSuccessRedraw)
            {
                redraw = true;
            }

            unsigned int pid = command->getSenderPid();
            if (commandExecuteSuccess)
            {
                LOG_INFO("CommitCommand", "command=" << command->getString()
                      << " from sender, name=" << executor->getSenderName(pid)
                      << ", pid=" << pid << ", success+");
            }
            else
            {
                LOG_ERROR("CommitCommand", "command=" << command->getString()
                      << " from sender, name=" << executor->getSenderName(pid)
                      << ", pid=" << pid << ", failed-");
                std::string comm_name = command->getCommandName();
                if ( !comm_name.empty())
                {
                    if (needs_errmem_entry(comm_name))
                        log_to_errmem("LAYERMANAGER: CommitCommand: %s, from %s(pid:%d) failed\n",
                                    command->getString().c_str(),
                                    executor->getSenderName(pid),
                                    pid);
                }
                else
                    LOG_ERROR("CommitCommand", "command name not available");
            }
            delete command;
        }
    }
    clientCommandQueue.clear();

    ExecutionResult returnValue = ExecutionFailed;

    if (success)
    {
        if (redraw)
        {
            returnValue = ExecutionSuccessRedraw;
        }
        else
        {
            returnValue = ExecutionSuccess;
        }
    }
    else
    {
        if (redraw)
        {
            returnValue = ExecutionFailedRedraw;
        }
        else
        {
            returnValue = ExecutionFailed;
        }
    }
    return returnValue;
}

const std::string CommitCommand::getString()
{
    return "CommitCommand()";
}
